import unittest
from MoinMoin.util.header import Header


class HeaderParsingTests(unittest.TestCase):

    def testEmptyHeader(self):
        """ util.header: empty headers """
        header = Header(self.request, "Empty header")
        self.assertEqual(header.length(), 0)

    def testIgnoreComments(self):
        """ util.header: ignore comments """
        header = Header(self.request, "##\nBody")
        expected = "##\n"
        self.assertEqual(header.length(), len(expected))

    def testNoLineEnding(self):
        """ util.header: missing line ending """
        header = Header(self.request, "#pi value")
        expected = "#pi value"
        self.assertEqual(header.length(), len(expected))

    def testBreakOnFirstNonHeader(self):
        """ util.header: break on first non-header """
        header = Header(self.request,
                        "#pi value\nbreak here!\n#pi value\n")
        expected = "#pi value\n"
        self.assertEqual(header.length(), len(expected))

    def testBreakOnInvalidPI(self):
        """ util.header: break on empty (invalid) pi """
        header = Header(self.request,
                        "#pi value\n# break here!\n#pi value\n")
        expected = "#pi value\n"
        self.assertEqual(header.length(), len(expected))

    def testProcessingInstruction(self):
        """ util.header: parse processing instruction """
        header = Header(self.request,
                        "## Comment\n#pi value\n#pragma key value\nBody\n")
        self.assertEqual(header.get('pi'), 'value')

    def testEmptyProcessingInstruction(self):
        """ util.header: empty processing instruction ignored """
        header = Header(self.request, "#pi   \nBody\n")
        self.assertEqual(header.get('pi'), None)
   
    def testPragma(self):
        """ util.header: parse processing instruction """
        header = Header(self.request,
                        "## Comment\n#pi value\n#pragma key value\nBody\n")
        self.assertEqual(header.get('pragma').get('key'), 'value')

    def testEmptyPragma(self):
        """ util.header: empty pragma ignored """
        header = Header(self.request, "#pragma key    \nBody\n")
        self.assertEqual(header.get('pragma'), None)
        
    def testStripWhitespace(self):
        """ util.header: strip white space around tokens """
        header = Header(self.request,
                        "#pi  value  \n#pragma  key  value  \nBody\n")
        self.assertEqual(header.get('pi'), 'value')
        self.assertEqual(header.get('pragma').get('key'), 'value')

