# -*- coding: utf-8 -*-
'''
MoinMoin 1.6-1.9 - SeeSaw Macro
    @copyright: 2008,2009,2011,2013 Jim Wight
    @licence: GNU GPL, see COPYING for details

PURPOSE
    To enable sections of a page to be see-sawed (toggled) between hidden
    and shown, or unconditionally placed in either state.

DESCRIPTION
    SeeSaw creates a control (link or button) for performing its actions.  It
    has two parts: at any time, one is shown and the other hidden. SeeSaw
    uses the CSS 'display' attribute to switch between the two parts, and to
    hide and show sections. Every page element that SeeSaw affects is normally
    identified with the CSS class name 'seesaw', so that it is unlikely for it
    to accidentally affect unrelated elements.

    The control is created by a call of the form:

        <<SeeSaw(section="section", toshow="toshow", tohide="tohide",
                 show=True|False, bg="background", inline="inline",
                 image="arrow|plumin", speed="slow|normal|fast"|integer,
                 seesaw=True|False, addclass="list", type="link|button",
                 effect="slide|fade")>>
    where

        section in basic usage - extended usage is described below - 'section'
                specifies the name of a section that the control will toggle;
                the name is attached to the parts of the control as a CSS class
                name, and needs to be attached to sections that the control is
                to affect; the default is 'section'
                 
         toshow specifies the text to display for showing the section; 
                the default is 'Show', except for inline sections where it
                is '»»'
                 
         tohide specifies the text to display for hiding the section;
                the default is 'Hide', except for inline sections where it
                is '««'
                
           show specifies whether the section is to be shown initially;
                the default is False
                 
             bg specifies the a background colour to be applied to the section;
                the default is None
                 
         inline specifies the text of an inline section; SeeSaw creates inline
                sections itself, immediately after the controls;
                the default is None
                 
          image selects an image set to use for the control instead of text,
                with toshow and tohide being ignored; 
                the default is None;
                
          speed specifies the rate at which the section should appear or 
                disappear, or provides a number of milliseconds for an
                animation to last;
                the default is None
                 
         seesaw SeeSaw will, by default, only toggle sections created for it,
                i.e. bearing the CSS class name 'seesaw'; this option, when
                set to False, enables non-SeeSaw elements to be toggled from a
                SeeSaw control;
                the default is True
                 
       addclass specifies a space-separated list of additional CSS class names
                to be associated with the control, and inline section, if any,
                to enable them to be manipulated using those names in other
                SeeSaw calls;
                the default is None

           type specifies the type of HTML element to create for the control;
                the default is 'link'

         effect specifies an alternative style of animation for showing and
                hiding the section; 
                the default is None

    (NB String arguments don't need to be quoted unless they contain spaces or
        special characters)
                 
    The (leading) arguments can also be given positionally, in the order of
    the keyword arguments above, e.g.

        <<SeeSaw(section1, Show, Hide, True)>>
        
    is equivalent to
    
        <<SeeSaw(section=section1, toshow=Show, tohide=Hide, show=True)>>

EXTENDED DESCRIPTION
 section and addclass
    Sections are typically set up as follows:

        {{{#!wiki seesaw section   or   {{{#!wiki seesaw/section
        }}}                             }}}

    where the word 'section' matches the value "section" in the corresponding
    SeeSaw call. This creates an HTML div with the classes 'seesaw' and
    'section' applied. In general, a section can be any HTML element to which
    CSS classes can be applied via MoinMoin markup, e.g. divs as just
    described, tables via tableclass (for whole tables), rowclass (for rows)
    and class (for cells), or others via {{{#!html. See also 'inline' below.

    In extended usage, 'section' in the SeeSaw call is a space- or
    slash-separated list of plain or prefixed section or CSS class names
    specifying actions to be carried out when the control is clicked. The
    items can have any of the following forms:

        section  - for toggling itself and other controls and sections
                   identified by "section"
        %section - for toggling controls and sections identified by "section"
        +section - for unconditionally putting controls and sections
                   identified by "section" into their shown state
        -section - for unconditionally putting controls and sections
                   identified by "section" into their hidden state

    If there is more than one plain name, the second and subsequent are treated
    as additional CSS class names (as if added by 'addclass') to be associated
    with the control, and inline section, if any, to enable them to be
    manipulated using those names in other SeeSaw calls. For example,

        <<SeeSaw("s1 s2 all")>>

    would toggle sections identified by "s1" and its control parts would also
    be toggled by calls toggling sections identified by either "s2" or
    "all". It is equivalent to

        <<SeeSaw(s1,addclass="s2 all")>> and <<SeeSaw("s1 s2",addclass=all)>>

    The order of priority, from highest to lowest, is toggles, shows and hides,
    so it is possible, for example, to show a section that is also hidden by
    virtue of a classification in the same sequence, e.g "+s1 -all" would
    result in s1 being shown even if also covered by 'all'.
    
    If the section argument doesn't contain a plain name, then the control has
    a single state and always displays the 'toshow' part (or equivalent for
    images); the 'show' argument is not relevant.

    If you want a section to be manipulated by extra class names, i.e.
    additional plain section names or names appearing in 'addclass' arguments,
    then those names should also be added to the sections, e.g. given

        <<SeeSaw("a both")>> <<SeeSaw("b,addclass=both)>> <<SeeSaw("%both")>>

        {{{#!wiki seesaw a both    and   {{{#!wiki seesaw b both

    the first SeeSaw call would create a control toggling its own parts and the
    first section, the second call would do the same for the second section,
    and the third would toggle both sections and the parts of both controls
    (keeping them in sync with the states of the sections).

 toshow, tohide and show
    By default, sections are hidden initially, but can be shown by adding
    'show' to the relevant markup. 'show' should be set to True in the
    matching SeeSaw call if its 'tohide' and 'toshow' arguments are different
    (so that the correct one can be shown initially).

    'toshow' and 'tohide' can accommodate text surrounding the control, to
    allow it to be different in the two cases. The three parts are
    distinguished by enclosing the text of the control between '<<' and '>>',
    e.g. toshow="<<Open>> me up", tohide="Now <<close>> me down". The middle
    part is ignored if 'image' is used, so can be empty.

    The toshow and tohide defaults only apply if both options are not set. If
    only one is set the other is given its value.
                 
 bg
    If a background colour is specified for a section, '"section"-bg'
    needs to be added to the corresponding markup to have the colour
    applied to the section, e.g.

        <<SeeSaw(section1, bg="#FFFFDD")>>

        ||<class="seesaw section1 section1-bg>Data||

    If there are multiple SeeSaw calls for the same section name, it is
    sufficient to use 'bg' in just one of them, with the last taking
    precedence if multiple, but different, values are given.

 inline

    The text of inline sections is embedded in SeeSaw calls. By default, inline
    sections are hidden initially, and the effects of 'show' and 'bg', and the
    addition of the 'section' and 'addclass' classes are handled directly by
    SeeSaw. Inline sections are created as HTML spans. Spans not adjacent to
    controls can be created by using the 3rd-party span macro, available at
    http://moinmo.in/MacroMarket/span.

 image
    Additional image sets can be made available as embedded data or from files.
    Instructions are in the 'Configuration' section of this file (below).

 seesaw 
 
   By setting the option 'seesaw' to False it is possible to apply SeeSaw to
    non-SeeSaw sections, e.g. the div created by the TableOfContents macro. In
    such a case, the section name in the SeeSaw call should be a class already
    attached to the div, or the class can be added as an extra name or by use
    of the addclass argument. If the section is shown initially (more than
    likely), 'show=True' should be set in order to get the control's text right
    - or the values of 'toshow' and 'tohide' can be reversed.

    When 'seesaw' is True (the default), the control affects only sections with
    'seesaw' and any of the names in the section argument attached as class
    names. When 'seesaw' is False, the control affects all sections with any
    names in the section argument attached as class names. In this case,
    sections will be shown initially.

    It is the presence of 'seesaw' on a section that causes it to be hidden
    initially. Thus, treating user-created sections as non-SeeSaw sections
    (without the addition of 'seesaw') is an option if most are to be shown
    initially.

 speed
    By default, sections are shown and hidden instantaneously. However, if a
    speed is specified, they are gradually revealed and withdrawn via an
    animation which adjusts their width, height and opacity simultaneously.
    The speed is applied to all sections specified in the section argument and
    is independent of the speeds applied to those sections in other calls,
    e.g. given

        <<SeeSaw(s1)>>   <<SeeSaw(s2, speed=fast)>> 
        <<SeeSaw("s3 %s1 %s2", speed=slow)>>

    both s1 and s2, as well as s3, will be animated slowly by the third call.

 effect
    By default, sections are shown and hidden instantaneously. The 'effect'
    option provides animations which are smoother in operation. The effect is
    applied to all sections specified in the section argument and is
    independent of the effects applied to those sections in other calls, as
    above for speed.

EXAMPLES
    Download seesawexamples.tgz from http://moinmo.in/Macromarket/SeeSaw

AUTHOR
    Jim Wight <j.k.wight@gmail.com>

HISTORY
    [v1.3] 2013-03-31
    Treat additional plain section names as added classes
    Embed image data
    Encapsulate control in span with different class from sections
    Use HTML custom data to pass bg
    Remove claim re coexistence with MoinMoin comments
    [Applicable to all HTML element types (JavaScript)]

    [v1.2] Not released
    As 1.3 but with different interepretation of effect with multiple sections

    [v1.1] 2011-01-23
    Add effect argument

    [v1.0] 2009-04-11
    Add support for multiple sections
    Add button type

    [v0.5] 2009-01-31
    Add seesaw option
    Remove unnecessary HTML if {pre|post}/{show|hide} not used
    Remove examples

    [v0.4] Not released
    Add speed option

    [v0.3] 2008-07-05
    Add image links

    [v0.2] 2008-05-12
    Accommodate different text surrounding the link

    [v0.1] 2008-05-01
    Initial version
'''


#--- Configuration ----------------------------------------
# 
# Change this to extend the list of image sets, either as base64
# encodings or as filenames ('show' image first, in both cases) e.g.
#
moreimages = dict(
#    name1 = ["data:image/png;base64,...", "data:image/jpeg;base64,..."],
#    name2 = ["showname2.jpg", "hidename2.gif"]
)
#
# Change this to the relative location of additional image sets
# provided as filenames, e.g.
#
# imageprefix="/moin_static19x/common"
imageprefix = "/img"
#
#
#--- End of Configuration ---------------------------------

Dependencies = []

def execute(macro, args):
    from MoinMoin import wikiutil
    import re

    def escape(arg):
        if arg is None:
            return arg
        else:
            return wikiutil.escape(arg)

    images = dict(
arrow = [
'''data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAkAAAAJCAYAAADgkQYQAAAAU0lEQVQYlZ2QIQ6AQAwEJ9lcU4mt
v09g4OMIeASfwRSLAEoQ4yab3SUimpn1iGiZyR1I6kACq6S5khI4gEXS9CZd2SSNlZTA7u7Dr6Sy
0/O6Lz+d7FJ7hQrZj94AAAAASUVORK5CYII=''',
'''data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAkAAAAJCAYAAADgkQYQAAAAd0lEQVQYlXXPMQ7DIAwF0G/pe/GS
gQkW7polU47XA3RF4gy/C4loGyx5sHmyseWcvfdesYiU0hskC4AGQA/ZSBZIAoBjgQ5JgCS4e32Y
1ty93kgSzOyckZmd19uNImKbUURsf2j8bR9o/+rPxbj0RbIs0bX2t/cBi+KBCE326nsAAAAASUVO
RK5CYII='''],
plumin = [
'''data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAkAAAAJCAYAAADgkQYQAAAAL0lEQVQYlWNkYGD4z0AEIKToPxMx
mrApwgAsOEyAsRlxGY/BJ8o6bIoYsQkQDCcAU8AKB/g5WdcAAAAASUVORK5CYII=''',
'''data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAkAAAAJCAYAAADgkQYQAAAAKklEQVQYlWNkYGD4z0AEIKToPxMx
phCliIWAtYzoihgpso4oRYw43IICAElFBRBz9sLvAAAAAElFTkSuQmCC
''']
)
    images.update(moreimages)

    effects = ['default', 'slide', 'fade']

    parser = wikiutil.ParameterParser('%(section)s%(toshow)s%(tohide)s%(show)b%(bg)s%(inline)s%(image)s%(speed)s%(seesaw)b%(addclass)s%(type)s%(effect)s')
    (count, params) = parser.parse_parameters(args)
    (section, toshow, tohide, show, bgnd, inline, image, speed, seesaw, addclass, control, effect) = \
        (params[x] for x in ('section', 'toshow', 'tohide', 'show', 'bg', 'inline', 'image', 'speed', 'seesaw', 'addclass', 'type', 'effect'))

    if section is None:
        section = 'section'

    if tohide is None and toshow is None:
        if inline is None:
            toshow = 'Show'; tohide = 'Hide'
        else:
            toshow = u'»»';  tohide = u'««'
    elif tohide is None and toshow is not None:
        tohide = toshow
    elif toshow is None and tohide is not None:
        toshow = tohide

    if show is None:
        show = False

    if image is not None:
        imagepair = images.get(image)
        if imagepair is not None:
            if not imagepair[0].startswith("data:image"):
                imagepair = (imageprefix.rstrip("/") + "/" + x for x in imagepair)
            (toshow, tohide) = ('''<img src="%s" style="vertical-align:middle"/>''' % x for x in imagepair)

    if speed is None:
        speed = 0
    try:
        speed = int(speed)
    except ValueError:
        speed = "'%s'" % speed

    if seesaw is None or seesaw:
        seesaw = 'true'
    else:
        seesaw = 'false'

    if addclass is None:
        addclass = ""

    if control == 'button':
        opentag = closetag = 'button'
    else:
        opentag = 'a href=""'; closetag = 'a'

    if effect not in effects:
        effect = effects[0]
    if (effect == 'default'):
        effect = 'dflt'

    sections = ' '.join(section.strip().replace(' ', '/').split('/')).split()
    plain = [x.encode("iso-8859-1") for x in sections if not re.compile(r'^[%+-].*').match(x)]
    toggles = [x.lstrip('%').encode("iso-8859-1") for x in sections if x.startswith('%')]
    shows   = [x.lstrip('+').encode("iso-8859-1") for x in sections if x.startswith('+')]
    hides   = [x.lstrip('-').encode("iso-8859-1") for x in sections if x.startswith('-')]
    try:
        section = plain[0]
        sectionarg = '''{toggle:%s, show:%s, hide:%s}''' % ([section] + toggles, shows, hides)
        addclass += ' '.join(plain[1:])
    except IndexError:
        section = ""
        sectionarg = '''{toggle:%s, show:%s, hide:%s}''' % (toggles, shows, hides)

    regex = re.compile(r'(.*)<<(.*)>>(.*)')
    preshw = postshw = prehd = posthd = ""
    matches = regex.match(toshow)
    if matches:
        preshw, toshow, postshw = matches.groups()
    matches = regex.match(tohide)
    if matches:
        prehd, tohide, posthd = matches.groups()

    section, preshw, postshw, prehd, posthd, bgnd, inline = (escape(x) for x in (section, preshw, postshw, prehd, posthd, bgnd, inline))
    if image is None:
        toshow = escape(toshow); tohide = escape(tohide)

    if inline is None:
        inlinesection = ""
    else:
        inlineopts = ""
        if bgnd is not None:
            inlineopts += ''' %s-bg''' % section
        if show:
            inlineopts += " show"
        inlinesection = '''<span class="seesaw %(section)s %(addclass)s %(inlineopts)s">%(inline)s</span>''' % locals()

    showpart = 'show part'; hidepart = 'hide part'
    showstyle = '''style="display:inline"'''; hidestyle = '''style="display:none"'''
    if show and section:
        showstyle, hidestyle = hidestyle, showstyle

    preshow = postshow = prehide = posthide = ''
    prepost = '''<span class="%s" %s>%s</span>'''
    if preshw:  
        preshow  = prepost % (showpart, showstyle, preshw)
    if postshw:  
        postshow = prepost % (showpart, showstyle, postshw)
    showpart = '''<span class="%(showpart)s" %(showstyle)s>%(toshow)s</span>''' % locals()

    if section:
        if prehd:  
            prehide  = prepost % (hidepart, hidestyle, prehd)
        if posthd: 
            posthide = prepost % (hidepart, hidestyle, posthd)
        hidepart = '''<span class="%(hidepart)s" %(hidestyle)s>%(tohide)s</span>''' % locals()
        if bgnd is not None:
            bgdata = '''data-section="%(section)s" data-bg="%(bgnd)s"''' % locals()
        else:
            bgdata = ""
    else:
        hidepart = bgdata = ""

    html = '''
<span class="seesawc %(section)s %(addclass)s" %(bgdata)s>
%(preshow)s
%(prehide)s
<%(opentag)s onClick="seeSaw(%(sectionarg)s, '%(effect)s', %(speed)s, %(seesaw)s); return false;">
%(showpart)s
%(hidepart)s
</%(closetag)s>
%(postshow)s
%(posthide)s
</span>
%(inlinesection)s
''' % locals()

    return macro.formatter.rawHTML(html)
