# -*- coding: iso-8859-1 -*-
"""
    
    copyright: 2003-2006 MoinMoin:ThomasWaldmann
    copyright: 2007-2009 by Roger Haase - added changes for fixedleft theme
    License: GNU GPL

"""

from MoinMoin import wikiutil
from MoinMoin.Page import Page
from MoinMoin.theme import ThemeBase
import re

class Theme(ThemeBase): 
    """ 
    This is the fixedleft theme.  The htdocs/css and /img directories were copied from
    the Moin 1.8.3 modernized theme and modified as needed.
    """

    name = "fixedleft"
    
    # the icons dict from modernized theme
    _ = lambda x: x     # We don't have gettext at this moment, so we fake it
    icons = {
        # key         alt                        icon filename      w   h
        # FileAttach
        'attach':     ("%(attach_count)s",       "moin-attach.png",   16, 16),
        'info':       ("[INFO]",                 "moin-info.png",     16, 16),
        'attachimg':  (_("[ATTACH]"),            "attach.png",        32, 32),
        # RecentChanges
        'rss':        (_("[RSS]"),               "moin-rss.png",      16, 16),
        'deleted':    (_("[DELETED]"),           "moin-deleted.png",  16, 16),
        'updated':    (_("[UPDATED]"),           "moin-updated.png",  16, 16),
        'renamed':    (_("[RENAMED]"),           "moin-renamed.png",  16, 16),
        'conflict':   (_("[CONFLICT]"),          "moin-conflict.png", 16, 16),
        'new':        (_("[NEW]"),               "moin-new.png",      16, 16),
        'diffrc':     (_("[DIFF]"),              "moin-diff.png",     16, 16),
        # General
        'bottom':     (_("[BOTTOM]"),            "moin-bottom.png",   16, 16),
        'top':        (_("[TOP]"),               "moin-top.png",      16, 16),
        'www':        ("[WWW]",                  "moin-www.png",      16, 16),
        'mailto':     ("[MAILTO]",               "moin-email.png",    16, 16),
        'news':       ("[NEWS]",                 "moin-news.png",     16, 16),
        'telnet':     ("[TELNET]",               "moin-telnet.png",   16, 16),
        'ftp':        ("[FTP]",                  "moin-ftp.png",      16, 16),
        'file':       ("[FILE]",                 "moin-ftp.png",      16, 16),
        # search forms
        'searchbutton': ("[?]",                  "moin-search.png",   16, 16),
        'interwiki':  ("[%(wikitag)s]",          "moin-inter.png",    16, 16),

        # smileys (this is CONTENT, but good looking smileys depend on looking
        # adapted to the theme background color and theme style in general)
        #vvv    ==      vvv  this must be the same for GUI editor converter
        'X-(':        ("X-(",                    'angry.png',         16, 16),
        ':D':         (":D",                     'biggrin.png',       16, 16),
        '<:(':        ("<:(",                    'frown.png',         16, 16),
        ':o':         (":o",                     'redface.png',       16, 16),
        ':(':         (":(",                     'sad.png',           16, 16),
        ':)':         (":)",                     'smile.png',         16, 16),
        'B)':         ("B)",                     'smile2.png',        16, 16),
        ':))':        (":))",                    'smile3.png',        16, 16),
        ';)':         (";)",                     'smile4.png',        16, 16),
        '/!\\':       ("/!\\",                   'alert.png',         16, 16),
        '<!>':        ("<!>",                    'attention.png',     16, 16),
        '(!)':        ("(!)",                    'idea.png',          16, 16),
        ':-?':        (":-?",                    'tongue.png',        16, 16),
        ':\\':        (":\\",                    'ohwell.png',        16, 16),
        '>:>':        (">:>",                    'devil.png',         16, 16),
        '|)':         ("|)",                     'tired.png',         16, 16),
        ':-(':        (":-(",                    'sad.png',           16, 16),
        ':-)':        (":-)",                    'smile.png',         16, 16),
        'B-)':        ("B-)",                    'smile2.png',        16, 16),
        ':-))':       (":-))",                   'smile3.png',        16, 16),
        ';-)':        (";-)",                    'smile4.png',        16, 16),
        '|-)':        ("|-)",                    'tired.png',         16, 16),
        '(./)':       ("(./)",                   'checkmark.png',     16, 16),
        '{OK}':       ("{OK}",                   'thumbs-up.png',     16, 16),
        '{X}':        ("{X}",                    'icon-error.png',    16, 16),
        '{i}':        ("{i}",                    'icon-info.png',     16, 16),
        '{1}':        ("{1}",                    'prio1.png',         15, 13),
        '{2}':        ("{2}",                    'prio2.png',         15, 13),
        '{3}':        ("{3}",                    'prio3.png',         15, 13),
        '{*}':        ("{*}",                    'star_on.png',       16, 16),
        '{o}':        ("{o}",                    'star_off.png',      16, 16),
    }
    # end of icon copy from modernized theme
    # add fixedleft icons
    icons['newwindow'] =   (_("[NEW WINDOW]"), "newwindow.png",   16, 13)
    icons['close'] =   (_("[CLOSE]"), "close.png",   48, 16)
    icons['hide'] =   (_("[HIDE]"), "hide.png",   13, 13)
    icons['show'] =   (_("[SHOW]"), "show.png",   13, 13)
    icons['menu'] =   (_("[MENU]"), "menu.png",   13, 16)
    del _


        
    def title(self, d):
        """ Assemble the title (now using breadcrumbs)
        
        modifications:
            - make titles shorter to fit in narrow sidebar.
        
        @param d: parameter dictionary
        @rtype: string
        @return: title html
        """
        _ = self.request.getText
        content = []
        if d['title_text'] == d['page'].split_title(): 
            curpage = ''
            segments = d['page_name'].split('/') # was: title_text
            for s in segments[:-1]:
                ss = self.shortenPagename(s) # create a short name for sidebar
                curpage += s
                content.append("<li>%s</li>" % Page(self.request, curpage).link_to(self.request, ss)) #  use short name
                curpage += '/'
            link_text = self.shortenPagename(segments[-1]) 
            link_title = _('Click to do a full-text search for ', formatted=False) + d['page_name'] #  alt text includes name
            link_query = {
                'action': 'fullsearch',
                'value': 'linkto:"%s"' % d['page_name'],
                'context': '180',
            }
            # we dont use d['title_link'] any more, but make it ourselves:
            link = d['page'].link_to(self.request, link_text, querystr=link_query, title=link_title, css_class='backlink', rel='nofollow')
            content.append(('<li>%s</li>') % link)
        else:
            content.append('<li>%s</li>' % wikiutil.escape(d['title_text']))

        html = '''<ul id="pagelocation">%s</ul>''' % "".join(content)
        return html

    def navibar(self, d):
        """ Assemble the navibar

        modifications - The fixedleft sidebar should be as stable as possible, so
            - do not add current page to the navibar
            - do not add sister pages to the navibar

        Assemble the navibar

        @param d: parameter dictionary
        @rtype: unicode
        @return: navibar html
        """
        request = self.request
        found = {} # pages we found. prevent duplicates
        items = [] # navibar items
        item = u'<li class="%s">%s</li>'
        current = d['page_name']

        # Process config navi_bar
        if request.cfg.navi_bar:
            for text in request.cfg.navi_bar:
                pagename, link = self.splitNavilink(text)
                if pagename == current:
                    cls = 'wikilink current'
                else:
                    cls = 'wikilink'
                items.append(item % (cls, link))
                found[pagename] = 1

        # Add user links to wiki links, eliminating duplicates.
        userlinks = request.user.getQuickLinks()
        for text in userlinks:
            # Split text without localization, user knows what he wants
            pagename, link = self.splitNavilink(text, localize=0)
            if not pagename in found:
                if pagename == current:
                    cls = 'userlink current'
                else:
                    cls = 'userlink'
                items.append(item % (cls, link))
                found[pagename] = 1

        # Assemble html
        items = u''.join(items)
        html = u'''<ul id="navibar">%s</ul>''' % items
        return html

    def trail(self, d):
        """ Assemble page trail
        
        modifications
            - trail is reversed to add most recent on top and delete old items from bottom
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: trail html
        """
        request = self.request
        user = request.user
        html = ''
        if not user.valid or user.show_page_trail:
            trail = user.getTrail()
            if trail:
                trail.reverse() # pages will drop off the bottom, not the top
                items = []
                for pagename in trail:
                    try:
                        interwiki, page = wikiutil.split_interwiki(pagename)
                        if interwiki != request.cfg.interwikiname and interwiki != 'Self':
                            link = (self.request.formatter.interwikilink(True, interwiki, page) +
                                    self.shortenPagename(page) +
                                    self.request.formatter.interwikilink(False, interwiki, page))
                            items.append('<li>%s</li>' % link)
                            continue
                        else:
                            pagename = page
                    except ValueError:
                        pass
                    page = Page(request, pagename)
                    title = page.split_title()
                    title = self.shortenPagename(title)
                    link = page.link_to(request, title)
                    items.append('<li>%s</li>' % link)
                html = '''<ul id="pagetrail">%s</ul>''' % ''.join(items)
        return html

    def html_head(self, d):
        """ Assemble html head
        
        modifications
            - Add javascript source file and favicon to whatever ThemeBase creates. 
            - insert a link to skip wiki navigation for screen readers
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted header
        """
        head = ThemeBase.html_head(self, d)
        head += u'\n<script type="text/javascript" src="%s/%s/js/%s.js"></script>' % (self.cfg.url_prefix_static, self.name, self.name)
        # add sortable feature -- use like this:   ||<tableclass="sortable">head1||head2||head3||
        head += u'\n<script type="text/javascript" src="%s/%s/js/sorttable.js"></script>' % (self.cfg.url_prefix_static, self.name)
        favicon = self.request.getPragma('favicon')
        if favicon:
            t = favicon.rindex(".")
            head += u'\n<link rel="icon" href="%s" type="%s">' % (favicon,"image/%s" % favicon[t + 1:])
        # insert a link to skip wiki navigation for screen readers
        head = head.replace(u'</title>',
            u''''</title>\n<link rel="Alternate" title="Wiki Page Content" href="#MainWikiPageContent">\n''',1)
        return head

    def header(self, d):
        """ Assemble page header

        Default behavior is to start a page div. Sub class and add
        footer items.
        
        modifications:
            - write out all wiki navigation into the sidebar 
        
        @param d: parameter dictionary
        @rtype: string
        @return: page header html
        """
        _ = self.request.getText

        html = [
            # Sidebar
            u'<div id="sidebar">', 
            self.hideSidebar(), # icon to hide sidebar
            u'<h1 id="wikinavigationheader">%s</h1>' % _(u'Wiki Navigation'), # hidden by common.css
            self.logoName(d),
            u'<div id="sidebarWikiMenu">',
            u'<ul id="wikiNavMenu">', 
            self.searchpanel(d),
            self.pageLinkTo(d),
            self.navipanel(d),
            self.pagepanel(d),
            self.userpanel(d),
            self.trailpanel(d),
            u'</ul>',
            u'</div>', #sidebarWikiMenu
            # add thispage class name to all links pointing to the page currently loading, used to make current page name bold
            u'<script type="text/javascript" language="javascript">linksToThisPage()</script>',
            u'</div>', # end sidebar div

            # wiki page
            u'<div id="wikipagecontent">',  # wikipagecontent div ends are emitted by footer
            self.showSidebar(d), # creates wiki navigation for sideways mode
            u'<script type="text/javascript" language="javascript">repeatHide()</script>', # hide sidebar based on cookie
            u'<h1 id="wikipageheader">%s</h1>' % _(u'Wiki Page Content'), # hidden by common.css
            u'<a name="MainWikiPageContent"></a>', # accessibility anchor for screen readers
            # on this theme there is nothing to separate header1 from header2
            self.emit_custom_html(self.cfg.page_header1),
            self.emit_custom_html(self.cfg.page_header2),
            self.msg(d),
            self.startPage(),
            ]

        return u''.join(html)
        
        
    def hideSidebar(self):
        """
        Return a html fragment with a hide button for the sidebar.
        """
        return u'''<img src="%s/%s/img/hide.png" id="hideIcon" width="13" height="13" 
            title="hide sidebar" onClick="hideSidebar(this);">''' % (self.cfg.url_prefix_static, self.name) 

    def showSidebar(self,d):
        """
        Return a html fragment with a show icon and an icon for a sideways drop down menu.
        """
        #~ editbarFixup = self.editbar(d).replace('More Actions &gt;','More Actions')
        html = [
            u'<div id="altWikiNavMenu">',
            self.sidewaysIcons(),
            u'</li></ul></div>',
            ]
        return '\n'.join(html)
        
    def sidewaysIcons(self):
        """Return html fragment to display show and menu icons."""
        html = [
            u'''<img src="%s/%s/img/show.png" id="showIcon" width="13" height="13" 
title="show sidebar" onClick="showSidebar();">''' % (self.cfg.url_prefix_static, self.name),
            u'''<ul class="sideways" id="menuIcon"><li><a href="#" id="contentWikiMenu">
<img src="%s/%s/img/menu.png"  width="13" height="16" ></a>''' % (self.cfg.url_prefix_static, self.name),
            ]
        return ''.join(html)

    def editorheader(self, d):
        """
        Assemble editor header as a left sidebar.
        
        modifications:
            - write out all wiki navigation into the sidebar 
            - write out custom hints
            
        @param d: parameter dictionary
        @rtype: string
        @return: page header html
        """
        _ = self.request.getText

        html = [
            # Sidebar
            u'<div id="sidebar">',
            self.hideSidebar(), # icon for hiding sidebar
            u'<h1 id="wikinavigationheader">%s</h1>' % _(u'Wiki Navigation'), # hidden by common.css
            self.logoName(d),
            u'<div id="sidebarWikiMenu">',
            u'<ul id="wikiNavMenu">', 
            self.pageLinkTo(d),
            self.editbuttons(d),
            self.edithelp(d),
            self.edit_hints(d),
            u'</ul>',
            u'</div>', # end sidebar
            u'</div>', #sidebarWikiMenu
            u'<div id="wikipagecontent">', # wikipagecontent div end is emitted by footer
            self.showSidebar(d), # creates wiki navigation for sideways mode
            u'<script type="text/javascript" language="javascript">repeatHide()</script>', # hide sidebar based on cookie
            u'<h1 id="wikipageheader">%s</h1>' % _(u'Wiki Page Content'), # hidden by common.css
            u'<a name="MainWikiPageContent"></a>', # accessibility anchor for screen readers
            self.emit_custom_html(self.cfg.page_header1),
            self.emit_custom_html(self.cfg.page_header2),
            self.msg(d),
            self.startPage(),
            ]
        return u'\n'.join(html)
        
    def footer(self, d, **keywords):
        """ Assemble page footer

        Default behavior is to end page div. Sub class and add
        footer items.
        
        modifications:
         - removed edit bar on bottom of page (is in sidepanel)
         - inserted end of div for wikipagecontent
         - added hack to work around IE7 bug
         - added javascript to delete trivial checkbox above edit textarea for IE6
        
        @param d: parameter dictionary
        @keyword ...:...
        @rtype: string
        @return: page footer html
        """
        page = d['page']
        html = [
            # End of page
            self.pageinfo(page),
            self.endPage(),
            
            # Pre footer custom html (not recommended!)
            self.emit_custom_html(self.cfg.page_footer1),
            
            # Footer
            u'<div id="footer">',
            self.credits(d),
            self.showversion(d, **keywords),
            u'</div>',
            
            # Post footer custom html
            self.emit_custom_html(self.cfg.page_footer2),

            # the following is a workaround for an IE7 bug,  on short pages with only text, or pages with only text showing when scrolled to page bottom,
            # IE7 hides the text.  The text can be viewed by selecting the text or forcing the page to be redrawn.
            # Or, adding this invisible image makes IE7 happy!  
            # @@@ should make_icon or make_iconlink be used below?
            u'<img src="%s/%s/img/newwindow.png" width="0" height="0" >' % (self.cfg.url_prefix_static, self.name), # this is fix for IE7 disappearing text
            # end of ie7 fix
            
            u'</div>',  # wikipagecontent
            
            # get rid of unwanted chkTrivial check box at top of edit area
            # must use javascript because IE6 does not understand CSS adjacent sibling selector
            u'<script type="text/javascript" language="javascript">delChkTrivialTop()</script>',
            ]
        return u'\n'.join(html)

    def editbuttons(self,d):
        """Place the input buttons for the text editor. These are outside the form element. 
        
        The tickle function will click the corresponding form element button.  
        Buttons can not tell the difference between a tickle and a click :-).
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: html formatted edit buttons
        """
        _ = self.request.getText
        mouseOver = '''onmouseover="this.style.cursor='pointer'" onmouseout="this.style.cursor='auto'"'''
        # put out GUI button if user preferences allow it
        if self.cfg.editor_default == 'text' and self.cfg.editor_force:
            guiButton = ''
        else:
            guiButton = '''<li id="xswitch2gui" ><a class="xbutton" name="xbutton_switch" onClick="tickle(this);" %s >GUI/Text Mode</a></li>\n''' % mouseOver
        html = [
            '<li class="sidepanel">%s<ul>' % ( _("Editor Controls")),
            # @@@ to eliminate the missing comment reminder, change checkComment to tickle on the line below
            u'''<li><a class="xbutton"  name="xbutton_save" onClick="checkComment(this);" %s >Save Changes</a></li>\n
<li><a class="xbutton" name="xbutton_preview"  onClick="tickle(this);" %s >Preview</a></li>\n
%s
<li  style="display: none;"><a class="xbutton" name="xbutton_load_draft" onClick="tickle(this);"  %s >Load Draft</a></li>\n
<li><a class="xbutton" name="xbutton_spellcheck" onClick="tickle(this);" %s >Check Spelling</a></li>\n
<li><a class="xbutton" name="xbutton_cancel" onClick="tickle(this);" %s >Cancel</a></li>\n
''' % (mouseOver, mouseOver, guiButton, mouseOver, mouseOver, mouseOver),
            u'</ul></li>',
            ]
        return u'\n'.join(html)

    def editpopup(self,hyperlink,windowName='popup'):
        """Return a hyperlink that will open a popup window.  Precede link with popup icon.
        
        The incoming hyperlink will look something like:
           <a href="/mywiki/HelpOnEditing">HelpOnEditing</a>
        Add an info icon and additional parameters  inside the "a" tag so a new window is opened.
        
        @param hyperink: string with valid anchor
        @rtype: unicode
        @return: modified anchor
        """
        # editor help windows are opened in new window, user is encouraged to read the help and close the window
        # todo: is there a way to show all the HelpOn pages in the sidebar menu when the user opens a help window?
        newParms =  u''' onclick="window.open(this.href,'%s','toolbar=no,location=no,directories=no,status=no,menubar=no,scrollbars=yes,copyhistory=no,resizable=yes,left=0,top=0').focus(); event.returnValue=false; return false;" 
onkeypress="window.open(this.href,'%s','toolbar=no,location=no,directories=no,status=no,menubar=no,scrollbars=yes,copyhistory=no,resizable=yes,left=0,top=0').focus(); event.returnValue=false; return false;" >
<img src="%s/%s/img/newwindow.png" 
border="0" width="16" height="13" alt="New window" title="New window">''' % (windowName,windowName, self.cfg.url_prefix_static, self.name)
        return '<li>%s</li>' % hyperlink.replace('>',newParms,1)
           
    def edithelp(self,d):
        """Add a panel of help links.   To avoid losing edits, open links in a new window.
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted help 
        """
        _ = self.request.getText
        html = [
            u'<li class="sidepanel">%s<ul>' % ( _("Editor Help")),
            self.editpopup(_("HelpOnEditing", wiki=True),windowName='hoe'),
            self.editpopup(_("HelpOnMoinWikiSyntax", wiki=True),windowName='sr'),
            u'</ul></li>',
            ]
        return u'\n'.join(html)
        
    def hintline(self,s,divclass='hint'):
        """Return a formatted hint example.
        
        @param s: a hint line
        @rtype: unicode
        @return: formatted hint line
        """
        return u'<li class="%s">%s</li>' % (divclass,s)

    def edit_hints(self,d):
        """Add a panel of syntax quick references for new users.
        
        We can not use the editor_quickhelp defined in wikiconfig.py or multiconfig.py without 
        effecting other themes, so define a new version here that will fit in a narrow column.
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: html formatted edit hints
        """
        _ = self.request.getText
        html = [
            u'<li class="sidepanel">%s<ul>' % (_("Editor Hints")),
            self.hintline(_("Text Formatting"),divclass='hintheader'),
            self.hintline("''<span class='hintital'>%s</span>''" % _('italics')),
            self.hintline("'''<span class='hintbold'>%s</span>'''" % _('bold')),
            self.hintline("'''''<span class='hintboit'>%s</span>'''''" % _('bold italics')),
            self.hintline("`<tt class='backtick'>%s</tt>`" % _('monospace')),
            
            self.hintline(_("Horizontal Rules"),divclass='hintheader'),
            self.hintline("---- %s" % _('(use 4 to 10 hyphens)')),
            
            self.hintline(_("Headings"),divclass='hintheader'),
            self.hintline("== %s 2 == %s" % (_("Title"),_("(use 1 to 5 = chars)"))),
            #~ self.hintline("===== %s 5 =====" % _("Title")),
            
            self.hintline(_("Lists"),divclass='hintheader'),
            self.hintline( _("Start with space(s):")),
            self.hintline("&nbsp;* %s" % _("round bullet")),
            self.hintline("&nbsp;1 %s" % _("numbered bullet")),
            #~ self.hintline("&nbsp;I %s" % _("Roman bullet")),
            self.hintline("&nbsp;i %s" % _("roman bullet")),
            self.hintline("&nbsp;A %s" % _("Alpha bullet")),
            #~ self.hintline("&nbsp;a %s" % _("alpha bullet")),
            
            self.hintline(_("Hyperlinks"),divclass='hintheader'),
            self.hintline(_("JoinCapitalizedWords")),
            self.hintline("http://www.myorg.org"),
            #~ self.hintline("[[http://www.myorg.org]]"),
            self.hintline("[[http://myorg.org|%s]]" % _("my site")),
            self.hintline("user@example.com"),
            
            self.hintline(_("Not a Hyperlink"),divclass='hintheader'),
            self.hintline(_("!JoinCapitalizedWords")),

            self.hintline(_("Embedding"),divclass='hintheader'),
            self.hintline(_("{{attachment:animage.jpg}}")),
            self.hintline(_("{{http://myorg.org/pix.png}}")),
            
            self.hintline(_("Tables"),divclass='hintheader'),
            self.hintline("||%s 1||%s 2||" % (_("cell"),_("cell"))),
            self.hintline("||||%s||" % _("2-column cell")),
            
            self.hintline(_("Macros"),divclass='hintheader'),
            self.hintline("&lt;&lt;%s&gt;&gt;" % _("TableOfContents")),
            self.hintline("&lt;&lt;FootNote(%s)&gt;&gt;" % _("some text")),
            self.hintline("&lt;&lt;Include(%s)&gt;&gt;" % _("SomePage")),
            self.hintline("&lt;&lt;BR&gt;&gt; - %s" % _("hard line break")),
            
            self.hintline(_("Variables"),divclass='hintheader'),
            self.hintline("@SIG@ - %s" % _("signature with time")),
            #~ self.hintline("@USER@ - %s" % _("signature")),
            u'</ul></li>',
            ]
        return u'\n'.join(html)

    def navipanel(self, d):
        """ Create a navigation panel.
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted navibar
        """
        _ = self.request.getText
        html = [
            u'<li class="sidepanel">%s' % ( _("Navigation Links")),
            self.navibar(d),
            u'</li>',
            ]
        return u''.join(html)

    def pagepanel(self, d):
        """ Create page actions panel . 
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted actions panel
        """
        _ = self.request.getText
        if self.shouldShowEditbar(d['page']):
            html = [
                u'<li class="sidepanel">%s' % ( _("Page Actions")),
                self.editbar(d),
                u'</li>',
                ]
            return u''.join(html)
        return ''   
        
    def userpanel(self, d):
        """ Create user panel.
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted user panel
        """
        _ = self.request.getText
        html = [
            u'<li class="sidepanel">%s' % (  _("User Actions")),
            self.username(d),
            u'</li>'
            ]
        return u''.join(html)
        
    def trailpanel(self, d):
        """ Create page trail panel.

        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted page trail panle
        """
        _ = self.request.getText
        trail = self.trail(d)
        if trail:
            html = [
                u'<li class="sidepanel">%s' % ( _("Page Trail")),
                trail,
                u'</li>'
                ]
        else:
            html = []
        return u''.join(html)

    def searchpanel(self,d):
        """Create search panel.
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted search panel
        """
        _ = self.request.getText
        data = {'iconurl': self.cfg.url_prefix_static,
            'theme_name': self.name,}
        html = [
            u'<li class="sidepanel" id="sidebarSearch">',
            u'''<a href="#" id="searchFormClose" onClick="return hideSearchForm()">
<img src="%(iconurl)s/%(theme_name)s/img/close.png" title="close"></a>''' % data,
            u'%s' % (_("Search Wiki")),
            self.searchform(d),
            u'</li>',
            u'<li id="sidewaysSearch"><a href="#" onClick="showSearchForm()">Search &gt;</a></li> ',
            ]
        return u''.join(html)
        
    def searchform(self, d):
        """
        assemble HTML code for the search forms
        
        modifications:
            - wrap title and fullsearch buttons in a div to position under search text box

        @param d: parameter dictionary
        @rtype: unicode
        @return: search form html
        """
        _ = self.request.getText
        form = self.request.form
        updates = {
            'search_label': _('Search:'),
            'search_value': wikiutil.escape(form.get('value', [''])[0], 1),
            'search_full_label': _('Text'),
            'search_title_label': _('Titles'),
            'baseurl': self.request.getScriptname(),
            'pagename_quoted': wikiutil.quoteWikinameURL(d['page'].page_name),
            'mouseOver': ''' ''',
            }
        d.update(updates)

        html = u'''
<form id="searchform" method="get" action="%(baseurl)s/%(pagename_quoted)s">
<div>
<input type="hidden" name="action" value="fullsearch">
<input type="hidden" name="context" value="180">
<label for="searchinput">%(search_label)s</label>
<input id="searchinput" type="text" name="value" value="%(search_value)s" size="20"
    onfocus="searchFocus(this)" onblur="searchBlur(this)"
    onkeyup="searchChange(this)" onchange="searchChange(this)" alt="Search">
<div id="searchbuttons">
<input id="titlesearch" name="titlesearch" type="submit"
    value="%(search_title_label)s" alt="Search Titles" %(mouseOver)s>
<input id="fullsearch" name="fullsearch" type="submit"
    value="%(search_full_label)s" alt="Search Full Text" %(mouseOver)s>
</div>
</div>
</form>
<script type="text/javascript">
<!--// Initialize search form
var f = document.getElementById('searchform');
f.getElementsByTagName('label')[0].style.display = 'none';
var e = document.getElementById('searchinput');
searchChange(e);
searchBlur(e);
//-->
</script>
''' % d
        return html
        
    def sitenameheader(self,d):
        """Create logo and sitename header for left panel.
        
        @param d: parameter dictionary
        @rtype: unicode
        @return: formatted logo and site name
        """
        _ = self.request.getText
        html = [self.logoName(d), self.pageLinkTo(d)]
        return u'\n'.join(html)
        
    def logoName (self,d):
        '''Return html fragment to display logo on sidepanel'''
        _ = self.request.getText
        html = [
            u'<div class="sidepanel">',
            self.logo().replace('></a></div>', ' title="%s"> %s</a></div>' % (self.cfg.page_front_page, self.cfg.sitename)),
            u'</div>'
            ]
        return u'\n'.join(html)

    def pageLinkTo (self, d):
        '''Return html fragment to display current page linkbacks on sidepanel.'''
        _ = self.request.getText
        html = [
            u'<li class="sidepanel" >%s' % (_("Current Page")),
            u'%s' % self.title(d), # was d["page_name"],
            u'</li>'
            ]
        return u''.join(html)

    def actionsMenu(self, page):
        """ Create actions menu list and items data dict

        The menu will contain the same items always, but items that are
        not available will be disabled (some broken browsers will let
        you select disabled options though).

        The menu should give best user experience for javascript
        enabled browsers, and acceptable behavior for those who prefer
        not to use Javascript.

        TODO: Move actionsMenuInit() into body onload - requires that the theme will render body,
              it is currently done in wikiutil/page.
              
        modifications from ThemeBase:
            - the More Actions drop down box is replaced with a popup list 
            - delete this entire method if you prefer drop down selection box

        @param page: current page, Page object
        @rtype: unicode
        @return: actions menu html fragment
        """
        request = self.request
        _ = request.getText
        rev = request.rev
        # rev was a hidden input within the more actions form
        # alternative will be to append it to anchor urls
        if rev: 
            revision = '&amp;rev=%s' % rev
        else:
            revision = '' 

        menu = [
            'raw',
            'print',
            'RenderAsDocbook',
            'refresh',
            '__separator__',
            'SpellCheck',
            'LikePages',
            'LocalSiteMap',
            '__separator__',
            'RenamePage',
            'CopyPage',
            'DeletePage',
            '__separator__',
            'MyPages',
            'SubscribeUser',
            '__separator__',
            'Despam',
            'revert',
            'PackagePages',
            'SyncPages',
            ]

        titles = {
            # action: menu title
            '__title__': _("More Actions:"),
            # Translation may need longer or shorter separator
            '__separator__': _('------------------------'),
            'raw': _('Raw Text'),
            'print': _('Print View'),
            'refresh': _('Delete Cache'),
            'SpellCheck': _('Check Spelling'), # rename action!
            'RenamePage': _('Rename Page'),
            'CopyPage': _('Copy Page'),
            'DeletePage': _('Delete Page'),
            'LikePages': _('Like Pages'),
            'LocalSiteMap': _('Local Site Map'),
            'MyPages': _('My Pages'),
            'SubscribeUser': _('Subscribe User'),
            'Despam': _('Remove Spam'),
            'revert': _('Revert to this revision'),
            'PackagePages': _('Package Pages'),
            'RenderAsDocbook': _('Render as Docbook'),
            'SyncPages': _('Sync Pages'),
            }

        options = []
        option = '<li><a href="%(baseurl)s?action=%(action)s%(revision)s">%(title)s</a></li>'
        disabledOption = '<li class="disabled">%(title)s</li>'
        
        # class="disabled" is a workaround for browsers that ignore
        # "disabled", e.g IE, Safari
        # for XHTML: data['disabled'] = ' disabled="disabled"'
        disabled = ' disabled class="disabled"'
        # @@@ is this best way to form link to current page?  Had trouble with subpages replicating parent page namd in url
        #  baseurl is full url as used here
        baseurl = self.request.getScriptname() + '/' + wikiutil.quoteWikinameURL(page.page_name) 

        # Format standard actions
        available = request.getAvailableActions(page)
        for action in menu:
            data = {'action': action, 'disabled': '', 'title': titles[action], 'baseurl': baseurl, 'revision': revision,}
            # removes excluded actions from the more actions menu
            if action in request.cfg.actions_excluded:
                continue

            # Enable delete cache only if page can use caching
            if action == 'refresh':
                if not page.canUseCache():
                    data['action'] = 'show'
                    data['disabled'] = disabled

            # revert action enabled only if user can revert
            if action == 'revert' and not request.user.may.revert(page.page_name):
                data['action'] = 'show'
                data['disabled'] = disabled

            # SubscribeUser action enabled only if user has admin rights
            if action == 'SubscribeUser' and not request.user.may.admin(page.page_name):
                data['action'] = 'show'
                data['disabled'] = disabled

            # PackagePages action only if user has write rights
            if action == 'PackagePages' and not request.user.may.write(page.page_name):
                data['action'] = 'show'
                data['disabled'] = disabled

            # Despam action enabled only for superusers
            if action == 'Despam' and not request.user.isSuperUser():
                data['action'] = 'show'
                data['disabled'] = disabled

            # Special menu items. Without javascript, executing will
            # just return to the page.
            if action.startswith('__'):
                data['action'] = 'show'

            # Actions which are not available for this wiki, user or page
            if (action == '__separator__' or
                (action[0].isupper() and not action in available)):
                data['disabled'] = disabled

            if data['disabled']:
                options.append(disabledOption % data)
            else:
                options.append(option % data)

        # Add custom actions not in the standard menu, except for
        # some actions like AttachFile (we have them on top level)
        more = [item for item in available if not item in titles and not item in ('AttachFile', )]
        more.sort()
        if more:
            # Add separator
            separator = disabledOption % {'action': 'show', 'disabled': disabled,
                                  'title': titles['__separator__'], 'baseurl': baseurl,} 
            options.append(separator)
            # Add more actions (all enabled) from /plugin/actions
            for action in more:
                data = {'action': action, 'disabled': '', 'baseurl': baseurl, 'revision': revision,} 
                title = action
                # Use translated version if available
                data['title'] = _(title)
                options.append(option % data)
        data = {
            'options': '\n'.join(options),
            'iconurl': self.cfg.url_prefix_static,
            'theme_name': self.name,
            }
        # finally, create the list -- needs href=xxx else will be made bold
        html = '''
<div class="moreactions">
<a href="moreactions" onClick="return showMoreActions()" >More Actions <span id="actionsGT">&gt;</span></a>
<ul id="moreActions">
<li id="hideMoreActions"><a href="#" onClick="return hideMoreActions()">
<img src="%(iconurl)s/%(theme_name)s/img/close.png" title="close"></a></li>
%(options)s
</ul></div>
''' % (data)

        return html


def execute(request):
    """
    Generate and return a theme object
        
    @param request: the request object
    @rtype: MoinTheme
    @return: Theme object
    """
    return Theme(request)

