# -*- coding: iso-8859-1 -*-
"""
    MoinMoin - SlideShow action

    Usage - Allows treatment of a wiki page as a set of slides.  Displays a
            single slide at a time, along with a navigation aid.

            A slide show page looks like this:

                general introduction or comments
                = Slide 1 title =
                Slide 1 contents
                = Slide 2 title =
                Slide 2 contents
                ...
                = Final slide title =
                Final slide contents

            The SlideShow action takes a parameter 'slidenumber', which is
            the (1-based) number of the slide to display.  The display uses
            the large screen 'projection' media.

            This action adds two sets of navigation aids for the slides:
            1) some javascript, so that mouse clicks or space bar move to the
               next page
            2) a navigation footer, similar to the [[Navigation]] macro

    History
    1.0 2005-03-25 First version
    1.1 2005-03-30 Revised thanks to some comments from NirSoffer.
                   Parse H1 tags instead of using the [[Slide]] macro.
                   Improve navigation.
    1.2 2005-03-31 Rewrite by NirSoffer: fix bug with heading within pre,
                   simplifiying code, auto correct slide number and much more.
                           
    @copyright: 2005 Jim Clark
    @license: GNU GPL, see COPYING for details.
"""

import re
import os
from MoinMoin import config, wikiutil
from MoinMoin.Page import Page

# TODO (long term): the page might use another markup, e.g rst
# Can we support any markup in a general way? 
from MoinMoin.parser.wiki import Parser

from MoinMoin.formatter.text_html import Formatter

# Must skip pre/code area when searching for headings
# This code will work with wiki markup only.
scanner = re.compile(r"""
    (?P<skip>{{{(?:.*\n)+?}}}) |                # skip this
    (?:^\s*(?P<heading>=\s(?P<text>.*)\s=)$)     # match this
""", re.MULTILINE | re.UNICODE | re.VERBOSE)


class SlideShow:

    name = 'SlideShow' # Used by action query

    # Seems that it does not work with Firefox when you use the find as
    # you type feature.
    script_template = """
<script type="text/javascript"><!--
function firstSlide() {window.location="%s"}
function prevSlide() {window.location="%s"}
function nextSlide() {window.location="%s"}
function lastSlide() {window.location="%s"}
function handleKey(e) {
    var key;
    if (e == null) {key = event.keyCode} //IE
    else {if (e.altKey || e.ctrlKey) {return true} key = e.which} // Mozilla
    switch(key) {
        case 49: firstSlide(); break //'1'
        case 32: nextSlide(); break  //space
        case 60: prevSlide(); break  //'<'
        case 62: nextSlide(); break  //'>'
        default:
    }
}
document.onkeypress = handleKey
//-->
</script>
"""

    def __init__(self, request, pagename):
        self.pagename = pagename

        # Start of suspected code
        # Is this realy needed? maybe formatter and page object already
        # exists?
        request.formatter = Formatter(request)
        request.page = Page(request, pagename)
        request.formatter.page = request.page
        self.request = request
        self.page = self.request.page
        self.formatter = self.request.formatter
        # End of suspected code
        
        # Order is important:
        # Get body -> get slide info -> normlize current slide number
        self.body = self.page.get_raw_body()
        self.slides = self.getSlides()
        try:
            self.current = int(request.form.get('n', [1])[0])
        except ValueError:
            self.current = 1
        self.current = self.normalizeSlideNumber(self.current)

    def normalizeSlideNumber(self, number):
        """ Keep number between 1 and last slide 
        
        Save one useless error message for the user.
        """
        number = max(1, number)
        number = min(number, len(self.slides))
        return number

    def getSlides(self):
        """ Create an index of slides
        
        slideinfo is a tuple of (title, start) representing each slide.
        The slides are separated by H1 markers.
        """
        slides = []
        title = ""
        start = 0
        for match in scanner.finditer(self.body):
            # Skip stuff
            if match.start('skip') != -1:
                continue
            
            # Save slide information
            start = match.start('heading')
            text = match.group('text').strip()
            slides.append((text, start))
            
        return slides

    def getSlideInfo(self, number):
        """ Return slide title and text """
        title, start = self.slides[number]
        try:
            end = self.slides[number + 1][1]
        except IndexError:
            end = len(self.body)
        text = self.body[start:end]
        
        return title, text

    def slideurl(self, slidenumber):
        """ Return a url for a link to another slide based on this page """
        return "%s/%s?action=%s&n=%d" % \
               (self.request.getScriptname(), 
               self.pagename,
               self.name,
               slidenumber)

    def navigationscript(self):
        """ Return a section of javascript for capturing mouse button and
        keyboard events and using for slide navigation.
        """
        urls = (self.slideurl(1),
                self.slideurl(self.current - 1),
                self.slideurl(self.current + 1),
                self.slideurl(len(self.slides) - 1))
        return (self.script_template % urls)

    # The navigation widget may be a spearate class, maybe we can build
    # a generic widget to be reused in other places.
            
    def _makeslidelink(self, i, text=None):
        """ Return a URL link to slide number 'i'.  The url title will be the
        slide number and the title attribute will be the slide title.
        
        TODO: this should use the same code in slideurl above.
        """
        if text is None:
            text = str(i)            
        if i == self.current:
            # Links to current slide are disabled
            return self.formatter.text(text)

        query = 'action=%s&n=%d' % (self.name, i)
        attrs = 'title="%s"' % self.slides[i-1][0]
        
        # TODO: use the formatter here? In other macros, using the
        # formatter was the correct solution, but I forgot why.
        return self.page.link_to(self.request, text=text,
                                 querystr=query, attrs=attrs)
    
    def listitem(self, i, text=None, css_class=None):
        """ Make list item for navigation """
        output = [
            self.formatter.listitem(1, css_class=css_class),
            self._makeslidelink(i, text),
            self.formatter.listitem(0),
            ]
        return ''.join(output)
        
    def navigation(self, maxSlides=15):
        """ Return slide navigation device 
        
        @param maxSlides 
        """
        current = self.current
        last = len(self.slides)

        # Setup a start and end range around the current slide - up to
        # maxSlides items will be displayed. Try to put the current
        # slide in the middle of the list of displayed slides.
        # e.g for current = 15 of 40 slides, with maxSlides = 11
        # |< << 10 11 12 13 14 [15] 16 17 18 19 20 >> >|
        start = max(1, current - maxSlides / 2)
        end = min(start + maxSlides, last)

        f = self.formatter
        output = [
            # Open div with list
            f.startContent("navigation"),
            f.bullet_list(1),
            
            # Link back to the parent page. TODO: use formatter instead
            # of page link?
            f.listitem(1),
            self.page.link_to(self.request, text='x', 
                              attrs='title="Stop Slide Show"'),
            f.listitem(0),
            
            # First and previous slides
            self.listitem(1, '|<'),
            self.listitem(max(current - 1, 1), '<<'),
        
            # Slides up to current
            ''.join([self.listitem(i) for i in range(start, current)]),
            
            # Current slide
            self.listitem(current, css_class='current'),

            # Slides after current to end
            ''.join([self.listitem(i) for i in range(current + 1, end)]),
            
            # Links to the next and last slides
            self.listitem(min(current + 1, last), '>>'), 
            self.listitem(last, '>|'),
            
            # Close list and div
            f.bullet_list(0),
            f.endContent(),
            ]
        return ''.join(output)

    def render(self):
        """ Render a slide from a page """
        request = self.request
        _ = request.getText
        
        request.http_headers()
        request.setContentLanguage(request.lang)
        
        # Send the header in 'projection' mode - for large print and no
        # headings. CSS handle page title. 
        wikiutil.send_title(request, self.page.split_title(request, force=1),
                            print_mode=1, media='projection')

        if not self.slides:
            # render page normally with an error message
            return "NotImplementedYet"

        # Get the requested slide text, and format it as a whole page,
        # along with the navigation aids.
        request.write(self.navigationscript())
        request.write(request.formatter.startContent("content"))
        title, text = self.getSlideInfo(self.current - 1)
        parser = Parser(text, request)
        parser.format(request.formatter)
        request.write(self.navigation())
        request.write(request.formatter.endContent())
        wikiutil.send_footer(request, self.pagename, print_mode=1)


def execute(pagename, request):
    SlideShow(request, pagename).render()
