# -*- coding: iso-8859-1 -*-
"""
    MoinMoin - ImprovedTableParser library

    @copyright: 2012, 2013, 2015 by Paul Boddie <paul@boddie.org.uk>
    @license: GNU GPL (v2 or later), see COPYING.txt for details.
"""

from MoinMoin import wikiutil
from MoinSupport import *
import re

__version__ = "0.2.1"

# Regular expressions.

syntax = {
    # At start of line:
    "rows"      : (r"^==(?!.*?==$)[ \t]?",      re.MULTILINE),          # == not-heading ws-excl-nl
    "continuations" : (r"^\s*\.\.(?!\.)[ \t]?", re.MULTILINE),          # .. ws-excl-nl or .. not-dot

    # Within text:
    "markers"   : (r"([{]{3,}|[}]{3,})",    re.MULTILINE | re.DOTALL),  # {{{... or }}}...
    "columns"   : (r"\|\|[ \t]*",           0),                         # || ws-excl-nl

    # At start of column text:
    "column"    : (r"^\s*<([^<].*?)>\s*(.*)",   re.DOTALL),             # ws < not-< attributes > ws
    }

patterns = {}
for name, (value, flags) in syntax.items():
    patterns[name] = re.compile(value, re.UNICODE | flags)

# Other regular expressions.

leading_number_regexp = re.compile(r"\d*")

# Constants.

up_arrow = u'\u2191'
down_arrow = u'\u2193'

# Functions.

def parse(s):

    "Parse 's', returning a table definition."

    table_attrs = {}
    rows = []

    # The following will be redefined upon the construction of the first row.

    row_attrs = {}
    columns = []
    columnnumber = 0

    # The following will be redefined upon the construction of the first column.

    column_attrs = {}

    # Process exposed text and sections.

    marker = None
    is_region = True

    # Initially, start a new row.

    row_continued = False

    for match_text in patterns["markers"].split(s):

        # Only look for table features in exposed text. Where a section is
        # defined, a marker will have been read and all regions before the
        # closing marker will not be exposed.

        if is_region and not marker:

            # Extract each row from the definition.

            for row_text in patterns["rows"].split(match_text):

                # Only create a new row when a boundary has been found.

                if not row_continued:

                    # Complete any existing row.

                    if columns:
                        extractAttributes(columns, row_attrs, table_attrs)
                        span_columns(columns, columnnumber)

                        # Replicate the last row to determine column usage.

                        column_usage = []

                        for column_attrs, text in columns:
                            try:
                                rowspan = int(column_attrs.get("rowspan", "1"))
                            except ValueError:
                                rowspan = 1
                            if rowspan > 1:
                                attrs = {}
                                attrs.update(column_attrs)
                                attrs["rowspan"] = str(rowspan - 1)
                                attrs["rowcontinuation"] = True
                                column_usage.append((attrs, text))
                            else:
                                column_usage.append(({}, None))

                        columns = column_usage

                    # Define a new collection of row attributes.

                    row_attrs = {}

                    # Reset the columns and make the list available for the
                    # addition of new columns, starting a new column
                    # immediately.

                    rows.append((row_attrs, columns))
                    column_continued = False
                    columnnumber = 0

                # Extract each column from the row.

                for text in patterns["columns"].split(row_text):

                    # Replace line continuation strings.

                    text = patterns["continuations"].sub("", text)

                    # Only create a new column when a boundary has been found.

                    if not column_continued:

                        # Complete any existing column.

                        if columns:
                            columnnumber = span_columns(columns, columnnumber)

                        # Extract the attribute and text sections.

                        match = patterns["column"].search(text)
                        if match:
                            attribute_text, text = match.groups()
                            column_attrs = parseAttributes(attribute_text, False)
                        else:
                            column_attrs = {}

                        # Define the new column with a mutable container
                        # permitting the extension of the text.

                        details = [column_attrs, text]

                        # Find the next gap in the columns.

                        while columnnumber != -1 and columnnumber < len(columns):
                            attrs, text = columns[columnnumber]
                            if text is None:
                                columns[columnnumber] = details
                                break
                            columnnumber += 1

                        # Or start adding at the end of the row.

                        else:
                            columnnumber = -1
                            columns.append(details)

                    else:
                        columns[columnnumber][1] += text

                    # Permit columns immediately following this one.

                    column_continued = False

                # Permit a continuation of the current column.

                column_continued = True

                # Permit rows immediately following this one.

                row_continued = False

            # Permit a continuation if the current row.

            row_continued = True

        else:

            # Handle section markers.

            if not is_region:

                # Interpret the given marker, closing the current section if the
                # given marker is the corresponding end marker for the current
                # section.

                if marker:
                    if match_text.startswith("}") and len(marker) == len(match_text):
                        marker = None

                # Without a current marker, start a section if an appropriate marker
                # is given.

                elif match_text.startswith("{"):
                    marker = match_text

            # Markers and section text are incorporated into the current column.

            columns[columnnumber][1] += match_text

        is_region = not is_region

    # Complete any final row.

    if columns:
        extractAttributes(columns, row_attrs, table_attrs)
        span_columns(columns, columnnumber)

    return table_attrs, rows

def span_columns(columns, columnnumber):

    """
    In the 'columns', make the column with the 'columnnumber' span the specified
    number of columns, returning the next appropriate column number.
    """

    column_attrs, text = columns[columnnumber]

    # Handle any previous column spanning other columns.

    if column_attrs.has_key("colspan"):
        try:
            colspan = int(column_attrs["colspan"])
        except ValueError:
            colspan = 1

        # Duplicate the current column as continuation
        # columns for as long as the colspan is defined.

        colspan -= 1
        while colspan > 0:
            attrs = {}
            attrs.update(column_attrs)
            attrs["colspan"] = str(colspan)
            attrs["colcontinuation"] = True

            if columnnumber != -1:
                columnnumber += 1
                if columnnumber < len(columns):
                    columns[columnnumber] = attrs, text
                else:
                    columnnumber = -1

            if columnnumber == -1:
                columns.append((attrs, text))

            colspan -= 1

    return columnnumber

def extractAttributes(columns, row_attrs, table_attrs):

    """
    Extract row- and table-level attributes from 'columns', storing them in
    'row_attrs' and 'table_attrs' respectively.
    """

    for column in columns:
        attrs = column[0]
        for name, value in attrs.items():
            if name.startswith("row") and name not in ("rowspan", "rowcontinuation"):
                row_attrs[name] = value
                del attrs[name]
            elif name.startswith("table"):
                table_attrs[name] = value
                del attrs[name]

# Sorting utilities.

def get_sort_columns(s, start=0):

    """
    Split the comma-separated string 's', extracting the column specifications
    of the form <column>["n"] where the suffix "n" indicates an optional
    numeric conversion for that column. Column indexes start from the specified
    'start' value (defaulting to 0).
    """

    sort_columns = []
    for column_spec in s.split(","):
        column_spec = column_spec.strip()

        ascending = True
        if column_spec.endswith("d"):
            column_spec = column_spec[:-1]
            ascending = False

        # Extract the conversion indicator and column index.
        # Ignore badly-specified columns.

        try:
            column = get_number(column_spec)
            suffix = column_spec[len(column):]
            fn = converters[suffix]
            sort_columns.append((max(0, int(column) - start), fn, ascending))
        except ValueError:
            pass

    return sort_columns

def get_column_types(sort_columns):

    """
    Return a dictionary mapping column indexes to conversion functions.
    """

    d = {}
    for column, fn, ascending in sort_columns:
        d[column] = fn, ascending
    return d

def get_number(s):

    "From 's', get any leading number."

    match = leading_number_regexp.match(s)
    if match:
        return match.group()
    else:
        return ""

def to_number(s, request):

    """
    Convert 's' to a number, discarding any non-numeric trailing data.
    Return an empty string if 's' is empty.
    """

    if s:
        return int(get_number(to_plain_text(s, request)))
    else:
        return s

def to_plain_text(s, request):

    "Convert 's' to plain text."

    fmt = getFormatterClass(request, "plain")(request)
    fmt.setPage(request.page)
    return formatText(s, request, fmt)

converters = {
    "n" : to_number,
    "" : to_plain_text,
    }

suffixes = {}
for key, value in converters.items():
    suffixes[value] = key

class Sorter:

    "A sorting helper class."

    def __init__(self, sort_columns, request):
        self.sort_columns = sort_columns
        self.request = request

    def __call__(self, row1, row2):
        row_attrs1, columns1 = row1
        row_attrs2, columns2 = row2

        # Apply the conversions to each column, comparing the results.

        for column, fn, ascending in self.sort_columns:
            column_attrs1, text1 = columns1[column]
            column_attrs2, text2 = columns2[column]

            # Ignore a column when a conversion is not possible.

            try:
                value1 = fn(text1, self.request)
                value2 = fn(text2, self.request)

                # Avoid empty strings appearing earlier than other values.

                if value1 == "" and value2 != "":
                    result = 1
                elif value1 != "" and value2 == "":
                    result = -1
                else:
                    result = cmp(value1, value2)

                # Where the columns differ, return a result observing the sense
                # (ascending or descending) of the comparison for the column.

                if result != 0:
                    return ascending and result or -result

            except ValueError:
                pass

        return 0

def write_sort_control(request, columnnumber, columns, sort_columns, column_types, table_name, start=0, write=None):

    """
    Using the 'request', write a sort control for the given 'columnnumber' in
    the collection of 'columns', using the existing 'sort_columns' and
    'column_types' to construct labels and links that modify the sort criteria,
    and using the given 'table_name' to parameterise the links.

    If the 'write' parameter is specified, use it to write output; otherwise,
    write output using the request.
    """

    fmt = request.formatter
    write = write or request.write
    _ = request.getText

    write(fmt.div(1, css_class="sortcolumns"))

    write(fmt.paragraph(1))
    write(fmt.text(_("Sort by columns...")))
    write(fmt.paragraph(0))

    # Start with the existing criteria without this column being involved.

    revised_sort_columns = [(column, fn, ascending)
        for (column, fn, ascending) in sort_columns if column != columnnumber]

    # Get the specification of this column.

    columnfn, columnascending = column_types.get(columnnumber, (to_plain_text, True))
    newsortcolumn = columnnumber, columnfn, columnascending
    newsortcolumn_reverse = columnnumber, columnfn, not columnascending
    newlabel = columns[columnnumber][1].strip()

    # Show this column in all possible places in the sorting criteria.

    write(fmt.number_list(1))

    just_had_this_column = False

    for i, (column, fn, ascending) in enumerate(sort_columns):
        new_sort_columns = revised_sort_columns[:]
        new_sort_columns.insert(i, newsortcolumn)
        label = columns[column][1].strip()

        arrow = columnascending and down_arrow or up_arrow
        arrow_reverse = not columnascending and down_arrow or up_arrow

        sortcolumns = get_sort_column_output(new_sort_columns)
        new_sort_columns[i] = newsortcolumn_reverse
        sortcolumns_reverse = get_sort_column_output(new_sort_columns)

        # Columns permitting the insertion of the selected column.

        if column != columnnumber and not just_had_this_column:
            write(fmt.listitem(1, css_class="sortcolumn"))

            # Pop-up element showing the column inserted before the sort column.

            write(fmt.span(1, css_class="sortcolumn-container"))
            write(fmt.span(1, css_class="newsortcolumn"))
            write(formatText(newlabel, request, fmt))

            write_sort_link(write, request, fmt, table_name, sortcolumns, arrow, "sortdirection")
            write_sort_link(write, request, fmt, table_name, sortcolumns_reverse, arrow_reverse, "sortdirection")

            write(fmt.span(0))
            write(fmt.span(0))

            # Link for selection of the modified sort criteria using the current
            # column and showing its particular direction.

            arrow = ascending and down_arrow or up_arrow
            arrow_reverse = not ascending and down_arrow or up_arrow
            write_sort_link(write, request, fmt, table_name, sortcolumns, u"%s %s" % (label, arrow), "")

        # Columns permitting removal or modification.

        else:
            write(fmt.listitem(1))

            # Either show the column without a link, since the column to be
            # inserted is already before the current column.

            if just_had_this_column:
                just_had_this_column = False
                arrow = ascending and down_arrow or up_arrow
                arrow_reverse = not ascending and down_arrow or up_arrow

                # Write the current column with its particular direction.

                write(fmt.span(1, css_class="unlinkedcolumn"))
                write(formatText(u"%s %s" % (label, arrow), request, fmt))
                write(fmt.span(0))

            # Or show the column with a link for its removal.

            else:
                just_had_this_column = True
                sortcolumns_revised = get_sort_column_output(revised_sort_columns)
                write_sort_link(write, request, fmt, table_name, sortcolumns_revised, u"%s %s" % (label, arrow), "removecolumn")

                # Alternative sort direction.

                write_sort_link(write, request, fmt, table_name, sortcolumns_reverse, arrow_reverse, "altdirection")

        write(fmt.listitem(0))

    if not just_had_this_column:

        # Write the sorting criteria with this column at the end.

        new_sort_columns = revised_sort_columns[:]
        new_sort_columns.append(newsortcolumn)

        sortcolumns = get_sort_column_output(new_sort_columns)
        new_sort_columns[-1] = newsortcolumn_reverse
        sortcolumns_reverse = get_sort_column_output(new_sort_columns)

        arrow = columnascending and down_arrow or up_arrow
        arrow_reverse = not columnascending and down_arrow or up_arrow

        write(fmt.listitem(1, css_class="appendcolumn"))

        # Pop-up element showing the column inserted before the sort column.

        write(fmt.span(1, css_class="newsortcolumn"))
        write_sort_link(write, request, fmt, table_name, sortcolumns, newlabel, "")
        write_sort_link(write, request, fmt, table_name, sortcolumns, arrow, "sortdirection")
        write_sort_link(write, request, fmt, table_name, sortcolumns_reverse, arrow_reverse, "sortdirection")
        write(fmt.span(0))

        write(fmt.listitem(0))

    write(fmt.number_list(0))

    write(fmt.div(0))

def write_sort_link(write, request, fmt, table_name, sortcolumns, label, css_class):

    "Write a link expressing sort criteria."

    write(fmt.url(1, "?%s#%s" % (
        wikiutil.makeQueryString("%s-sortcolumns=%s" % (table_name, sortcolumns)),
        fmt.qualify_id(fmt.sanitize_to_id(table_name))
        ), css_class=css_class))
    write(formatText(label, request, fmt))
    write(fmt.url(0))

def get_sort_column_output(columns, start=0):

    "Return the output criteria for the given 'columns' indexed from 'start'."

    return ",".join([("%d%s%s" % (column + start, suffixes[fn], not ascending and "d" or ""))
        for (column, fn, ascending) in columns])

# Common formatting functions.

def formatTable(text, request, fmt, attrs=None, write=None):

    """
    Format the given 'text' using the specified 'request' and formatter 'fmt'.
    The optional 'attrs' can be used to control the presentation of the table.

    If the 'write' parameter is specified, use it to write output; otherwise,
    write output using the request.
    """

    # Parse the table region.

    table_attrs, table = parse(text)

    # Define the table name and an anchor attribute.

    table_name = attrs.get("name")
    if table_name:
        table_attrs["tableid"] = table_name
    else:
        table_name = table_attrs.get("tableid")

    # Only attempt to offer sorting capabilities if a table name is specified.

    if table_name:

        # Get the underlying column types.

        column_types = get_column_types(get_sort_columns(attrs.get("columntypes", "")))

        # Get sorting criteria from the region.

        region_sortcolumns = attrs.get("sortcolumns", "")

        # Update the column types from the sort criteria.

        column_types.update(get_column_types(get_sort_columns(region_sortcolumns)))

        # Determine the applicable sort criteria using the request.

        sortcolumns = getQualifiedParameter(request, table_name, "sortcolumns")
        if sortcolumns is None:
            sortcolumns = region_sortcolumns

        # Define the final sort criteria.

        sort_columns = get_sort_columns(sortcolumns)
        data_start = int(getQualifiedParameter(request, table_name, "headers") or attrs.get("headers", "1"))

        # Update the column types from the final sort criteria.

        column_types.update(get_column_types(sort_columns))

        # Sort the rows according to the values in each of the specified columns.

        if sort_columns:
            headers = table[:data_start]
            data = table[data_start:]

            # Perform the sort and reconstruct the table.

            sorter = Sorter(sort_columns, request)
            data.sort(cmp=sorter)
            table = headers + data

    # Otherwise, indicate that no sorting is being performed.

    else:
        sort_columns = None

    # Write the table.

    write = write or request.write
    write(fmt.table(1, table_attrs))

    for rownumber, (row_attrs, columns) in enumerate(table):
        write(fmt.table_row(1, row_attrs))
        sortable_heading = sort_columns is not None and rownumber == data_start - 1

        for columnnumber, (column_attrs, column_text) in enumerate(columns):

            # Always skip column continuation cells.

            if column_attrs.get("colcontinuation"):
                continue

            # Where sorting has not occurred, preserve rowspans and do not write
            # cells that continue a rowspan.

            if not sort_columns:
                if column_attrs.get("rowcontinuation"):
                    continue

            # Where sorting has occurred, replicate cell contents and remove any
            # rowspans.

            else:
                if column_attrs.has_key("rowspan"):
                    del column_attrs["rowspan"]

            # Remove any continuation attributes that still apply.

            if column_attrs.has_key("rowcontinuation"):
                del column_attrs["rowcontinuation"]

            write(fmt.table_cell(1, column_attrs))

            if sortable_heading:
                write(fmt.div(1, css_class="sortablecolumn"))

            write(formatText(column_text or "", request, fmt))

            # Add sorting controls, if appropriate.

            if sortable_heading:
                write_sort_control(request, columnnumber, columns, sort_columns, column_types, table_name, write=write)
                write(fmt.div(0))

            write(fmt.table_cell(0))

        write(fmt.table_row(0))

    write(fmt.table(0))

def formatTableForOutputType(text, request, mimetype, attrs=None, write=None):

    """
    Format the given 'text' using the specified 'request' for the given output
    'mimetype'.

    The optional 'attrs' can be used to control the presentation of the table.

    If the 'write' parameter is specified, use it to write output; otherwise,
    write output using the request.
    """

    write = write or request.write

    if mimetype == "text/html":
        write('<html>')
        write('<body>')
        fmt = request.html_formatter
        fmt.setPage(request.page)
        formatTable(text, request, fmt, attrs, write)
        write('</body>')
        write('</html>')

# vim: tabstop=4 expandtab shiftwidth=4
