# -*- coding: iso-8859-1 -*-
"""
    MoinMoin - Accessible side by side diff

    Please add for proper display following code to screen.css:
        .diff caption {
            font-weight: bold;
            text-align: left;
            margin-top: 1em;
        }
        .diff th {
            font-weight: bold;
            text-align: left;
        }
        /* Use the same here as .diff-added span */
        .diff-added ins {
            background-color: #80FF80;
        }
        /* Use the same here as .diff-removed span */
        .diff-removed del {
            background-color: #FFFF80;
        }
        /* Visually hidden information for screenreaders,
        i.e. is read but not displayed */
        .screenreader_info {
            position: absolute;
            top: -10000px;
            left: -10000px;
        }

    Add language support for:
        _('Line Number')
        _('Deletions')
        _('Additions')
        _('Show differences between page revisions')
        _('deleted')
        _('deleted end')
        _('added')
        _('added end')
        _('blank space')

    ToDo: Add ltr-support to css styles

    @copyright: 2007 by Oliver Siemoneit
    @copyright: 2002 by Jrgen Hermann <jh@web.de>
    @copyright: 2002 by Scott Moonen <smoonen@andstuff.org>
    @license: GNU GPL, see COPYING for details.
"""

from MoinMoin.support import difflib
from MoinMoin.wikiutil import escape

def _indent(line):
    eol = ''
    while line and line[0] == '\n':
        eol += '\n'
        line = line[1:]
    stripped = line.lstrip()
    if len(line) - len(stripped):
        line = "&nbsp;" * (len(line) - len(stripped)) + stripped
    return eol + line

def _prepare(pane, kw_begin, kw_end):
    import logging
    result = ''
    _pane = pane
    while _pane.find(kw_begin) != -1:
        start = _pane.find(kw_begin)
        end = _pane.find(kw_end)
        # Do only look for changes in a word, ignore longer text blocks
        if _pane[start:end].find(' ') != -1:
            result += _pane[:(end+len(kw_end))]
            _pane = _pane[(end+len(kw_end)):]
            logging.warning("Ignored text!!! result:|%s| _pane:|%s|" % (result, _pane))
            continue
        # Ignore added/deleted space blanks
        if _pane[start:end].find('&nbsp;') != -1:
            result += _pane[:(end+len(kw_end))]
            _pane = _pane[(end+len(kw_end)):]
            logging.warning("Ignored &nbsp;!!! result:|%s| _pane:|%s|" % (result, _pane))
            continue

        # Do the job: If only a character in a word is changed, mark the whole word
        # as changed
        
        # Split is buggy and fails catastrophically if separator cannot be found
        try:
            head, start_word = _pane[:start].rsplit(' ', 1)
            head += ' '
        except:
            head = ''
            start_word = _pane[:start]
        try:
            end_word, tail = _pane[start:].split(' ', 1)
            tail = ' ' + tail
        except:
            tail = ''
            end_word = _pane[start:]

        end_word = end_word.replace(kw_begin, '')
        end_word = end_word.replace(kw_end, '')

        result += (head + kw_begin + start_word + end_word + kw_end)
        _pane = tail
 
        logging.warning("pane: |%s|" % pane)
        logging.warning("head: |%s|" % head)
        logging.warning("start_word: |%s|" % start_word)
        logging.warning("end_word: |%s|" % end_word)
        logging.warning("tail: |%s|" % tail)
        logging.warning("result: |%s|" % result)

    result += _pane
    
    return result


# This code originally by Scott Moonen, used with permission.
def diff(request, old, new, caption):
    """ Find changes between old and new and return
        HTML markup marking them.
    """
    _ = request.getText

    seq1 = old.splitlines()
    seq2 = new.splitlines()

    seqobj = difflib.SequenceMatcher(None, seq1, seq2)
    linematch = seqobj.get_matching_blocks()

    if len(seq1) == len(seq2) and linematch[0] == (0, 0, len(seq1)):
        # No differences.
        return _("No differences found!")

    result = """
<table class="diff" summary="%(sum)s">
<caption>%(cap)s</caption>
<thead>
<tr>
<th id="r" abbr="">%(line)s</th> 
<th id="d" abbr="%(del)s:">%(del)s</th>
<th id="a" abbr="%(add)s:">%(add)s</th>
</tr>
</thead>
""" % { 'sum': _('Show differences between page revisions'),
        'cap': caption,
        'line': _('Line Number'),
        'del': _('Deletions'),
        'add': _('Additions'),}

    # Print all differences
    result += """<tbody>
"""
    lastmatch = (0, 0)

    # Start: iterate through lines
    for match in linematch:
        # Starts of pages identical?
        if lastmatch == match[0:2]:
            lastmatch = (match[0] + match[2], match[1] + match[2])
            continue

        # We have only one line number in AccessibleMoin showing the changed line of the original
        # version linking to the position of the changed line in the new version
        llineno, rlineno = lastmatch[0]+1, lastmatch[1]+1 
        t_line = _("Line") + " %d"
        result += """<tr> 
<td headers="r">%s</td>
""" % (request.formatter.line_anchorlink(1, rlineno) + request.formatter.text(t_line % llineno) + request.formatter.line_anchorlink(0))

        leftpane = ''
        rightpane = ''
        linecount = max(match[0] - lastmatch[0], match[1] - lastmatch[1])
        for line in range(linecount):
            if line < match[0] - lastmatch[0]:
                if line > 0:
                    leftpane += '\n'
                leftpane += seq1[lastmatch[0] + line]
            if line < match[1] - lastmatch[1]:
                if line > 0:
                    rightpane += '\n'
                rightpane += seq2[lastmatch[1] + line]

        charobj = difflib.SequenceMatcher(None, leftpane, rightpane)
        charmatch = charobj.get_matching_blocks()

        if charobj.ratio() < 0.5:
            # Insufficient similarity.
            if leftpane:
                leftresult = """<del>%s</del>""" % _indent(escape(leftpane))
            else:
                leftresult = ''

            if rightpane:
                rightresult = """<ins>%s</ins>""" % _indent(escape(rightpane))
            else:
                rightresult = ''
        else:
            # Some similarities; markup changes.
            charlast = (0, 0)

            leftresult = ''
            rightresult = ''
            for thismatch in charmatch:
                if thismatch[0] - charlast[0] != 0:
                    leftresult += """<del>%s</del>""" % _indent(
                        escape(leftpane[charlast[0]:thismatch[0]]))
                if thismatch[1] - charlast[1] != 0:
                    rightresult += """<ins>%s</ins>""" % _indent(
                        escape(rightpane[charlast[1]:thismatch[1]]))
                leftresult += escape(leftpane[thismatch[0]:thismatch[0] + thismatch[2]])
                rightresult += escape(rightpane[thismatch[1]:thismatch[1] + thismatch[2]])
                charlast = (thismatch[0] + thismatch[2], thismatch[1] + thismatch[2])

        leftpane = '<br>\n'.join(map(_indent, leftresult.splitlines()))
        rightpane = '<br>\n'.join(map(_indent, rightresult.splitlines()))

        # Prepare output for screenreaders

##        # If only a few characters have been deleted/added in a word, mark the whole word
##        # as deleted/added. Otherwise there is not chance for the blind to understand the
##        # changes having been made to the document.
##        leftpane = _prepare(leftpane, '<del>', '</del>')
##        rightpane = _prepare(rightpane, '<ins>', '</ins>')

        # Add some information for screenreaders which mark and explain the changes having been made
        leftpane = leftpane.replace('<del>', '<del><span class="screenreader_info">%s</span>' % _('deleted'))
        leftpane = leftpane.replace('</del>', '<span class="screenreader_info">%s</span></del>' % _('deleted end'))
        leftpane = leftpane.replace('&nbsp;', '&nbsp;<span class="screenreader_info">%s</span>' % _('blank space'))
        rightpane = rightpane.replace('<ins>', '<ins><span class="screenreader_info">%s</span>' % _('added'))
        rightpane = rightpane.replace('</ins>', '<span class="screenreader_info">%s</span></ins>' % _('added end'))
        rightpane = rightpane.replace('&nbsp;', '&nbsp;<span class="screenreader_info">%s</span>' % _('blank space'))
        
        
        result += """<td headers="d" class="diff-removed">
%s
</td>
<td headers="a" class="diff-added">
%s
</td>
</tr>
""" % (leftpane, rightpane)

        lastmatch = (match[0] + match[2], match[1] + match[2])
        # End: iterate through lines
        
    result += """</tbody>
</table>
"""
    return result

